/*
  This is a calibration routine for the Adafruit ISM330DHCX IMU

  5000 Snapshots of the Accelerometer X, Y, and Z Sensor is Taken
  with the data sent out over Bluetooth to an Arduino with microSD card.

  That Arduino saves the data as a csv file on the microSD card
*/

#include <Adafruit_ISM330DHCX.h>
#include <Adafruit_Sensor.h>

#define SAMPLERATE_DELAY_MS 10
#define NUMBER_CALIBRATION_READINGS 2500

Adafruit_ISM330DHCX ism330dhcx;
String imuName = "ISM330DHCX";

float imuPositive[NUMBER_CALIBRATION_READINGS];
float imuNegative[NUMBER_CALIBRATION_READINGS];

float xOffset, yOffset, zOffset;
float xScale, yScale, zScale;
unsigned long startTime;
unsigned long elapsedTime;
bool calibrated = false;
char axisToMeasure = 'X';
char polarityToMeasure = '+';


void setup() {
  Serial.begin(115200);
  while (!Serial);
  Wire.begin();

  Serial.print(imuName);
  Serial.println(" Calibration Program");

  initializeIMU();
  Serial.println("IMU Initialized");

  Wire.setClock(4000000);

  Serial.print("Place the IMU on the Calibration Platform with the (");
  Serial.print(polarityToMeasure);
  Serial.print(")");
  Serial.print(axisToMeasure);
  Serial.println("-Axis Up.");
  delay(2000);

  // Once All Offsets and Scales Have Be Calculated, End Loop
  do {
    //
    // Scan to See if User Has IMU Orriented Correctly for Recording the Axis Data
    //

    // Get Sensor Reading
    sensors_event_t accel;
    sensors_event_t gyro;
    sensors_event_t temp;
    ism330dhcx.getEvent(&accel, &gyro, &temp);

    if 
      (((accel.acceleration.x > 9) && (axisToMeasure == 'X') && (polarityToMeasure == '+')) || 
      ((accel.acceleration.x < -9) && (axisToMeasure == 'X') && (polarityToMeasure == '-')) || 
      ((accel.acceleration.y > 9) && (axisToMeasure == 'Y') && (polarityToMeasure == '+')) || 
      ((accel.acceleration.y < -9) && (axisToMeasure == 'Y') && (polarityToMeasure == '-')) || 
      ((accel.acceleration.z > 9) && (axisToMeasure == 'Z') && (polarityToMeasure == '+')) ||
      ((accel.acceleration.z < -9) && (axisToMeasure == 'Z') && (polarityToMeasure == '-'))) 
    {
      Serial.print("The IMU is Correctly Orriented, Calibration of the (");
      Serial.print(polarityToMeasure);
      Serial.print(")");
      Serial.print(axisToMeasure);
      Serial.print("-Axis will Begin in 5...");
      delay(1000);
      Serial.print("4...");
      delay(1000);
      Serial.print("3...");
      delay(1000);
      Serial.print("2...");
      delay(1000);
      Serial.println("1.");
      delay(1000);

      Serial.println("Recording...");
      recordIMUFloatData();
      if (polarityToMeasure == '-') {
        calculateOffsetsAndScales();
      }
      advanceToNextSensor();

    } else {
      Serial.print("Reorrient the IMU for (");
      Serial.print(polarityToMeasure);
      Serial.print(")");
      Serial.print(axisToMeasure);
      Serial.println("-Axis Calibration");
      delay(1500);
    }

  } while (calibrated == false);

  Serial.println("Write Down These Values:");
  Serial.println();
  Serial.print("X-Axis Offset: ");
  Serial.print(xOffset, 4);
  Serial.print(" Scale: ");
  Serial.println(xScale, 4);
  Serial.print("Y-Axis Offset: ");
  Serial.print(yOffset, 4);
  Serial.print(" Scale: ");
  Serial.println(yScale, 4);
  Serial.print("Z-Axis Offset: ");
  Serial.print(zOffset, 4);
  Serial.print(" Scale: ");
  Serial.println(zScale, 4);
}

void loop() {
}

void initializeIMU() {
  //
  // IMU Model Specific Initialization Code
  //
  if (!ism330dhcx.begin_I2C()) {
    Serial.println("Failed to initialize ISM330DHCX IMU");
    while (1);
  }
  // Initialise IMU Device
  Serial.print(imuName);
  Serial.print(" Accelerometer sample rate = ");
  switch (ism330dhcx.getAccelDataRate()) {
  case LSM6DS_RATE_SHUTDOWN:
    Serial.println("0 Hz");
    break;
  case LSM6DS_RATE_12_5_HZ:
    Serial.println("12.5 Hz");
    break;
  case LSM6DS_RATE_26_HZ:
    Serial.println("26 Hz");
    break;
  case LSM6DS_RATE_52_HZ:
    Serial.println("52 Hz");
    break;
  case LSM6DS_RATE_104_HZ:
    Serial.println("104 Hz");
    break;
  case LSM6DS_RATE_208_HZ:
    Serial.println("208 Hz");
    break;
  case LSM6DS_RATE_416_HZ:
    Serial.println("416 Hz");
    break;
  case LSM6DS_RATE_833_HZ:
    Serial.println("833 Hz");
    break;
  case LSM6DS_RATE_1_66K_HZ:
    Serial.println("1.66 KHz");
    break;
  case LSM6DS_RATE_3_33K_HZ:
    Serial.println("3.33 KHz");
    break;
  case LSM6DS_RATE_6_66K_HZ:
    Serial.println("6.66 KHz");
    break;
  }
  Serial.println();
  Serial.println("Acceleration in Meters/second2");
  Serial.println("X\tY\tZ");
  //
  // End IMU Model Specific Initialization Code
  //
}

void recordIMUFloatData() {
  for (int i = 0; i < NUMBER_CALIBRATION_READINGS; i++) {
    // Record the start time
    startTime = millis();

    //
    // IMU Model Specific Code
    //
    sensors_event_t accel;
    sensors_event_t gyro;
    sensors_event_t temp;
    ism330dhcx.getEvent(&accel, &gyro, &temp);
    switch (axisToMeasure) {
      case 'X':
        if (polarityToMeasure == '+') {
          imuPositive[i] = accel.acceleration.x;
        } else {
          imuNegative[i] = accel.acceleration.x;
        }
        break;
      case 'Y':
        if (polarityToMeasure == '+') {
          imuPositive[i] = accel.acceleration.y;
        } else {
          imuNegative[i] = accel.acceleration.y;
        }
        break;
      case 'Z':
        if (polarityToMeasure == '+') {
          imuPositive[i] = accel.acceleration.z;
        } else {
          imuNegative[i] = accel.acceleration.z;
        }
        break;
      default:
        break;
    }
    //
    // End IMU Model Specific Code
    //

    // Calculate how long the loop took to execute
    elapsedTime = millis() - startTime;
    // Delay for the remaining time to ensure 10ms per loop iteration, if necessary
    if (elapsedTime < SAMPLERATE_DELAY_MS) {
      delay(SAMPLERATE_DELAY_MS - elapsedTime);
    }
  }
  Serial.print("(");
  Serial.print(polarityToMeasure);
  Serial.print(")");
  Serial.print(axisToMeasure);
  Serial.println("-Axis Data Recorded");
}

void calculateOffsetsAndScales() {
  float imuPositiveTotal, imuNegativeTotal = 0;

  //
  // Calculate Offsets and Scales
  //
  for (int i = 0; i < NUMBER_CALIBRATION_READINGS; i++) {
    imuPositiveTotal += imuPositive[i];
    imuNegativeTotal += imuNegative[i];
  }

  float imuPositiveMean = (imuPositiveTotal / NUMBER_CALIBRATION_READINGS);
  float imuNegativeMean = (imuNegativeTotal / NUMBER_CALIBRATION_READINGS);

  switch (axisToMeasure) {
    case 'X':
      xOffset = ((imuPositiveMean + imuNegativeMean) / 2);
      xScale = ((imuPositiveMean - imuNegativeMean) / 2);
      break;
    case 'Y':
      yOffset = ((imuPositiveMean + imuNegativeMean) / 2);
      yScale = ((imuPositiveMean - imuNegativeMean) / 2);
      break;
    case 'Z':
      zOffset = ((imuPositiveMean + imuNegativeMean) / 2);
      zScale = ((imuPositiveMean - imuNegativeMean) / 2);
      break;
  }
}

void advanceToNextSensor() {
  // Advance to the Next Polarity or Axis to Record IMU Data
  if (polarityToMeasure == '+') {
    polarityToMeasure = '-';
  } else if (axisToMeasure == 'X') {
    axisToMeasure = 'Y';
    polarityToMeasure = '+';
  } else if (axisToMeasure == 'Y') {
    axisToMeasure = 'Z';
    polarityToMeasure = '+';
  } else if (axisToMeasure == 'Z') {
    calibrated = true;
  }
}